function [out,fmin,parout]=comd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,set,startx,par)
% COMD employs the co-mirror descent method for solving the problem min{f(x): g(x)<=0, x in X}.
% f is the "objective function", g is the "constraint function" and X is the "underlying  set".
%
% Underlying assumptions:
% f and the components of g are convex Lipschitz functions
% ====================================================
% Usage:
% out               = COMD(Ffun,Ffun_grad,Gfun,Gfun_grad,set,startx,[par])
% [out,fmin]        = COMD(Ffun,Ffun_grad,Gfun,Gfun_grad,set,startx,[par])
% [out,fmin,parout] = COMD(Ffun,Ffun_grad,Gfun,Gfun_grad,set,startx,[par])
% ====================================================
% Input:
% Ffun        - function handle for the objective function f
% Ffun_sgrad  - function handle for the subgradient of the objective function f
% Gfun        - function handle for the constraint function g
% Gfun_sgrad  - function handle for the subgradients of the components of the constraint function (arranged in columns)
% set         - an underlying set; allowed values: 'simplex', 'ball','box','spectahedron'
% startx      - a starting vector
% par         - a struct which contains different values required for the operation of COMD
% Fields of par:
%     Fields relevant for all sets:
%
%     max_iter   - maximal number of iterations [default: 1000]
%     eco_flag   - true if economic version (without calculating objective function
%                              values) should run, otherwise false [default: false]
%     print_flag - true if internal printing should take place, otherwise false [default: true]
%     stuck_iter - maximal allowed number of iterations without improvement [default: 10000]
%     feas_eps   - feasibility tolerance
%
%     Fields relevant for 'simplex': (underlying set {x: sum(x) [= or <=] simplexR, lowb<=x<=upb})
%                simplexR   - the radius of the simplex [default=1]
%                isequal    - true if equality is imposed, otherwise false  [default: true]
%                lowb       - lower bound on the variables (scalar/vector) [default: 0]
%                upb        - upper bound on the variables (scalar/vector) [default: inf]
%
%     Fields relevant for 'spectahedron': (underlying set {X: trace(X) [= or <=] specR, X psd, eig(X)<=upb})
%                specR      - value of required trace [default=1]
%                isequal    - true if equality is imposed, otherwise false  [default: true]
%                upb        - upper bound on the eigenvalues (scalar only) [default: inf]
%
%     Fields relevant for 'ball': (underlying set: {x: norm(x-ballCenter,'fro')<=ballR})
%                ballR      - radius of the ball [default: 1]
%                ballCenter - center of ball (same dimensions as startx) [default: zeros(size(startx))]
%
%     Fields relevant for 'box': (underlying set: {x: lowb <= x <= upb})
%                lowb       - lower bound on the variables (scalar/vector/matrix) [default: 0]
%                upb        - upper bound on the variables(scalar/vector/matrix) [default: 1]
%
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    consVec   - vector containing all the constraints violations quantities (when relevant)

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.


if ((nargin < 6)  || (isa (Ffun_sgrad,'function_handle') ~= 1) )
    error('usage: comd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,set,startx,[par])    ')
end

%checking whether the given value of set is allowed
if ((strcmp(set,'simplex') ~= 1)  && (strcmp(set,'ball') ~= 1)  && ( strcmp(set,'box') ~= 1 ) && ...
        ( strcmp(set,'spectahedron') ~= 1 ))
    error('usage: md(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,set,startx,[par]) - set should be one of the values simplex/ball/box/spectahedron') ;
end

%setting default values for par fields
if (nargin < 7)
    par =  struct();
end

if ( isempty(Ffun) || ~isa(Ffun,'function_handle') )
    if (nargout > 1)
        error('Cannot calculate fmin when Ffun is not a function handle')
    else
        if ( ~isfield (par,'eco_flag') || (~par.eco_flag))
            fprintf('Ffun is not a function handle, changing to economical mode\n') ;
            par.eco_flag = true ;
            par.print_flag = false ;
        end
    end
end

par  = initialize_par_for_md(par,set,startx) ;

printcon=true ;
if ((isempty(Gfun)) && (isempty(Gfun_sgrad)))
    if (par.print_flag)
        fprintf('Gfun and Gfun_sgrad are not given, using the zero function as constraint function \n') ;
    end
    printcon= false ;
    Gfun = @EmptyG ;
    Gfun_sgrad = @EmptyG_sgrad ;
else
    if ((isa (Gfun,'function_handle') ~= 1) || (isa (Gfun_sgrad,'function_handle') ~= 1 ))
        error('Gfun or Gfun_sgrad is not a function handle') ;
    end
end

if (strcmp(set,'ball') == 1)
    if (isinf(par.ballR) || any(any(isinf(par.ballCenter))))
        error('Finite parameters are required in par.ballR and par.ballCenter')
    end
elseif (strcmp (set, 'spectahedron') == 1)
    if ((~isscalar(par.upb)) || (~isscalar(par.specR)))
        error('Scalar parameters are required for par.upb and par.specR') ;
    elseif  (isinf(par.specR))
        %the parameters are scalar, verify  that they are finite
        error('Finite parameters are required in par.specR')
    end
elseif (strcmp (set, 'simplex') == 1)
    if (  any(any (isinf(par.lowb))))
        error('Finite parameters are required in par.lowb')
    end
else % box
    if (  any(any (isinf(par.lowb))) || any(any (isinf(par.upb))))
        error('Finite parameters are required in par.lowb annd par.upb')
    end
end
%checking proper dimension for input

if (strcmp(set,'spectahedron'))
    %check that input for spectaheron mode is a matrix
    if (~ismatrix(startx) || (size(startx,1) ~= size(startx,2)))
        error('Spectaheron mode run only on square matrices') ;
    end
elseif (strcmp (set,'simplex'))
    %check that input for simplex mode is a vector
    if (~isvector(startx))
        error('Simplex mode run only on vectors') ;
    end
    
end

%checking that the lower bound is not too big


if (strcmp(set,'simplex') == 1)
    
    %comparing lowb and simplexR
    if (isscalar(par.lowb) )
        sumlowb = length(startx) * par.lowb ; %sumlowb represents the sum of the lowb vector
    else
        sumlowb = sum(par.lowb) ;
    end
    
    if (sumlowb > par.simplexR)
        error('The lower bound is too big with respect to SimplexR') ;
    end
    
    %comparing upb and simplexR
    
    if (isscalar(par.upb) )
        sumupb = length(startx) * par.upb ; %sumupb represents the sum of the upb vector
    else
        sumupb = sum(par.upb) ;
    end
    
    if (sumupb < par.simplexR)
        error('The upper bound is too small with respect to SimplexR') ;
    end
end


%First printing
if(par.print_flag)
    fprintf('*********************\n');
    fprintf('Co-Mirror\n') ;
    fprintf('*********************\n')
    if (printcon)
        fprintf('#iter       \tfun. val.     \tcon. val.\n');
    else
        fprintf('#iter       \tfun. val.\n');
    end
    
end

%Running the required optimization program
if (strcmp(set,'simplex') == 1)
    [out,fmin,parout] =comirror_formd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,par.feas_eps,par.simplexR,par.max_iter,par.isequal,par.lowb,par.upb,par.eco_flag,par.print_flag,printcon,par.stuck_iter) ;
elseif (strcmp(set,'spectahedron') == 1)
    [out,fmin,parout] =comirror_specta_formd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,par.feas_eps,par.specR,par.max_iter,par.isequal,par.upb,par.eco_flag,par.print_flag,printcon,par.stuck_iter) ;
elseif (strcmp(set,'ball') == 1)
    [out,fmin,parout] =cosd_formd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,par.feas_eps,par.max_iter,true,par.ballCenter,0,par.ballR,par.eco_flag,par.print_flag,printcon,par.stuck_iter) ;
elseif   (strcmp(set,'box') == 1)
    [out,fmin,parout] =cosd_formd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,par.feas_eps,par.max_iter,false,0,par.lowb,par.upb,par.eco_flag,par.print_flag,printcon,par.stuck_iter) ;
end

if (par.eco_flag && (nargout > 1))
    %calculate the last function value
    fmin = Ffun(out) ;
end

if (par.print_flag)
    fprintf('----------------------------------\n') ;
    fprintf('Optimal value = %15f \n',fmin) ;
end



    function gt = EmptyG(~)
        gt = 0 ;
    end

    function der = EmptyG_sgrad(x)
        der = zeros(size(x)) ;
    end
end
